#ifndef __CBaseImage__
#define __CBaseImage__

#include "IImage.hpp"
using Exponent::GUI::Graphics::IImage;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Graphics
		{
			/**
			 * @class CBaseImage CBaseImage.hpp
			 * @brief A basic implementation of an image, including its object counting
			 *
			 * @date 29/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CBaseImage.hpp,v 1.3 2007/02/08 21:07:54 paul Exp $
			 */
			class CBaseImage : public IImage
			{

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * Construction
				 */
				CBaseImage();

				/**
				 * Destruction
				 */
				virtual ~CBaseImage();

//	===========================================================================

				/**
				 * Get the size of the image
				 * @retval const CDimension& The size of the image
				 */
				virtual const CDimension &getImageSize() const { return m_dimension; }

				/**
				 * Get the normalised image size
				 * @retval const CRect& 0,0 original size
				 */
				virtual const CRect &getNormalisedImageSize() const { return m_size; }

				/**
				 * Get the file name of the image
				 * @retval const CSystemString& The name of the file that is currently loaded
				 */
				virtual const CSystemString &getFileName() const { return m_filename; }

//	===========================================================================

				/**
				 * Load from a file
				 * @note This function is a stub and should be overridden, currently always returns false
				 * @param filename The name of the file
				 * @retval bool True if loaded the image properly, false on error
				 */
				virtual bool loadFromFile(const CSystemString &filename) { return false; }

				/**
				 * Unload the image
				 */
				virtual void unloadImage();

//	===========================================================================

				/**
				 * Does the image have an alpha channel
				 * @retval bool True if the image has an alpha channel, false otherwise
				 */
				virtual bool imageHasAlpha() const { return m_hasAlpha; }

				/**
				 * Is there an image loaded
				 * @retval bool True if image is loaded, false otherwise
				 */
				virtual bool hasImageLoaded() const { return m_imageLoaded; }

				/**
				 * Get the last error that occured
				 * @retval const CString& The last error
				 */
				const CString &getLastError() const { return m_lastError; }

//	===========================================================================

				#ifdef WIN32
					/**
					 * Get the image handle
					 * @retval HBITMAP The windows bitmap 
					 */
					virtual HBITMAP getImageHandle() const { return m_theBitmap; }
				#else
					/**
					 * Get the image handle
					 * @retval CGImageRef The mac image ref 
					 */
					virtual CGImageRef getImageHandle() const { return m_theBitmap; }
				#endif

//	===========================================================================

				/**
				 * Get a description of the object
				 * @param string On return is filled with the description
				 * @param size The size of the stirng
				 */
				virtual void getObjectDescription(char *string, const long size) const;

				/**
				 * Increment the reference count
				 */
				virtual void referenced();

				/**
				 * Decrement the reference count. Object is deleted if reference count is <= 0
				 */
				virtual void dereference();

				/**
				 * Get the reference count
				 * @retval long The reference count
				 */
				virtual long getReferenceCount() const;

//	===========================================================================

			protected:

//	===========================================================================

				bool m_hasAlpha;				/**< Do we have alpha ? */
				long m_referenceCount;			/**< How many things point to this item? */
				bool m_imageLoaded;				/**< Is there an image loaded? */
				CDimension m_dimension;			/**< dimension of the image */
				CRect m_size;					/**< size of the image */
				CSystemString m_filename;		/**< Path to the file loaded */

				CString m_lastError;			/**< The last error that occurred, fill this in to have it included in exception reports */

				#ifdef WIN32
					HBITMAP m_theBitmap;		/**< The image */
				#else
					CGImageRef m_theBitmap;		/**< The image */
				#endif

			};
		}
	}
}
#endif	// End of CBaseImage.hpp